/**
 * BundleLive Background Service Worker v2.0.1
 * Manages badge, relays messages, handles install defaults.
 * Auto-update: checks version + fetches remote config periodically.
 */

let totalSales = 0;
const CURRENT_VERSION = '3.8.0';
const BASE_URL = 'https://bundlelive.com';
const VERSION_CHECK_INTERVAL = 30 * 60 * 1000; // 30 minutes
const CONFIG_FETCH_INTERVAL = 15 * 60 * 1000; // 15 minutes

// ─── VERSION CHECK & AUTO-UPDATE ──────────────────────
async function checkForUpdate() {
  try {
    const resp = await fetch(`${BASE_URL}/api/extension/version`, { cache: 'no-cache' });
    if (!resp.ok) return;
    const data = await resp.json();
    if (data.version && data.version !== CURRENT_VERSION) {
      console.log(`[BL] 🔄 Update available: ${CURRENT_VERSION} → ${data.version}`);
      chrome.storage.local.set({
        extension_update_available: true,
        extension_latest_version: data.version
      });
      chrome.action.setBadgeText({ text: '↑' });
      chrome.action.setBadgeBackgroundColor({ color: '#fbbf24' });
    } else {
      chrome.storage.local.set({ extension_update_available: false });
    }
  } catch(e) {
    console.log('[BL] Version check failed:', e.message);
  }
}

// ─── FETCH REMOTE CONFIGS ─────────────────────────────
async function fetchRemoteConfigs() {
  try {
    const [captureResp, ordersResp] = await Promise.allSettled([
      fetch(`${BASE_URL}/extension/capture-config.json`, { cache: 'no-cache' }),
      fetch(`${BASE_URL}/extension/orders-config.json`, { cache: 'no-cache' })
    ]);
    if (captureResp.status === 'fulfilled' && captureResp.value.ok) {
      const config = await captureResp.value.json();
      chrome.storage.local.set({ captureConfig: config });
      console.log('[BL] 📡 Capture config updated');
    }
    if (ordersResp.status === 'fulfilled' && ordersResp.value.ok) {
      const config = await ordersResp.value.json();
      chrome.storage.local.set({ ordersConfig: config });
      console.log('[BL] 📡 Orders config updated');
    }
  } catch(e) {
    console.log('[BL] Config fetch failed:', e.message);
  }
}

// ─── SERVICE WORKER KEEP-ALIVE ────────────────────────
// Chrome kills service workers after ~30s of inactivity.
// Use a periodic alarm to keep it alive during active shows.
chrome.alarms.create('keepAlive', { periodInMinutes: 0.4 }); // every ~24s

// ─── PERIODIC CHECKS VIA ALARMS ───────────────────────
chrome.alarms.create('versionCheck', { periodInMinutes: 30 });
chrome.alarms.create('configFetch', { periodInMinutes: 15 });
chrome.alarms.create('jwtRefresh', { periodInMinutes: 720 }); // every 12 hours

// ─── AUTO JWT REFRESH ─────────────────────────────────
async function refreshJwt() {
  try {
    const { authToken, serverUrl } = await chrome.storage.local.get(['authToken', 'serverUrl']);
    if (!authToken || authToken === 'bundlelive2026') return;
    const url = (serverUrl || BASE_URL) + '/api/auth/refresh';
    const resp = await fetch(url, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json', 'Authorization': 'Bearer ' + authToken }
    });
    const data = await resp.json();
    if (data.success && data.token) {
      chrome.storage.local.set({ authToken: data.token });
      console.log('[BL] 🔑 JWT refreshed');
    }
  } catch(e) {
    console.log('[BL] JWT refresh failed:', e.message);
  }
}

chrome.alarms.onAlarm.addListener((alarm) => {
  if (alarm.name === 'keepAlive') {
    // Just waking up is enough to keep the service worker alive
    console.log('[BL] 💓 Keep-alive ping');
  }
  if (alarm.name === 'versionCheck') checkForUpdate();
  if (alarm.name === 'configFetch') fetchRemoteConfigs();
  if (alarm.name === 'jwtRefresh') refreshJwt();
});

// ─── SILENT PRINTING ─────────────────────────────────
async function silentPrint(html, width, height) {
  try {
    if (!chrome.printing) throw new Error('Printing API not available (requires ChromeOS or optional permission grant)');
    const printers = await chrome.printing.getPrinters();
    if (printers.length === 0) throw new Error('No printers found');
    
    const saved = await chrome.storage.local.get(['defaultPrinter']);
    const printer = saved.defaultPrinter 
      ? printers.find(p => p.id === saved.defaultPrinter) || printers[0]
      : printers[0];
    
    const printHtml = `<!DOCTYPE html>
      <html><head>
        <style>
          @page { size: ${width} ${height}; margin: 0; }
          html, body { margin: 0; padding: 0; width: ${width}; height: ${height}; }
        </style>
      </head><body>${html}</body></html>`;
    
    const blob = new Blob([printHtml], { type: 'text/html' });
    const url = URL.createObjectURL(blob);
    
    const ticket = {
      version: '1.0',
      print: {
        color: { type: 'STANDARD_MONOCHROME' },
        duplex: { type: 'NO_DUPLEX' },
        page_orientation: { type: 'PORTRAIT' },
        copies: { copies: 1 },
        media_size: {
          width_microns: Math.round(parseFloat(width) * 1000),
          height_microns: Math.round(parseFloat(height) * 1000),
        }
      }
    };
    
    await chrome.printing.submitJob({
      job: {
        printerId: printer.id,
        title: 'BundleLive Bin Label',
        ticket: ticket,
        contentType: 'text/html',
        document: url
      }
    });
    
    console.log('[BL Extension] 🖨️ Silent print sent to:', printer.name);
    URL.revokeObjectURL(url);
    return { success: true, printer: printer.name };
  } catch(e) {
    console.error('[BL Extension] Print error:', e);
    return { success: false, error: e.message };
  }
}

// Listen for messages from BundleLive dashboard (external)
chrome.runtime.onMessageExternal.addListener((msg, sender, sendResponse) => {
  if (msg.type === 'print-label') {
    silentPrint(msg.html, msg.width, msg.height).then(sendResponse);
    return true;
  }
  if (msg.type === 'get-printers') {
    chrome.printing.getPrinters().then(printers => {
      sendResponse({ success: true, printers: printers.map(p => ({ id: p.id, name: p.name })) });
    });
    return true;
  }
  if (msg.type === 'set-default-printer') {
    chrome.storage.local.set({ defaultPrinter: msg.printerId });
    sendResponse({ success: true });
    return true;
  }
});

chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
  if (msg.type === 'sale_captured') {
    totalSales = msg.count || totalSales + 1;
    // Only show sale count badge if no update is available
    chrome.storage.local.get(['extension_update_available'], (r) => {
      if (!r.extension_update_available) {
        chrome.action.setBadgeText({ text: String(totalSales) });
        chrome.action.setBadgeBackgroundColor({ color: '#4ade80' });
      }
    });
  }
  
  if (msg.type === 'connected') {
    chrome.action.setBadgeBackgroundColor({ color: '#4ade80' });
  }
  
  if (msg.type === 'disconnected') {
    chrome.action.setBadgeText({ text: '!' });
    chrome.action.setBadgeBackgroundColor({ color: '#ef4444' });
  }

  if (msg.type === 'get_total') {
    sendResponse({ totalSales });
    return true;
  }

  if (msg.type === 'get_auth_token') {
    chrome.storage.local.get(['authToken'], function(r) {
      sendResponse({ token: r.authToken || '' });
    });
    return true; // async response
  }

  if (msg.type === 'silent_print_label') {
    const { buyer, item, price, bin } = msg;
    const binLabel = bin || '';
    const labelHtml = `
      <div style="font-family:Arial,sans-serif;padding:4px;text-align:center;">
        <div style="font-size:28px;font-weight:bold;margin-bottom:2px;">${binLabel ? 'BIN ' + binLabel : 'BundleLive'}</div>
        <div style="font-size:14px;font-weight:bold;">@${buyer || 'unknown'}</div>
        <div style="font-size:11px;margin-top:2px;">${item || 'Live Sale'} — $${(price || 0).toFixed(2)}</div>
        <div style="font-size:9px;color:#666;margin-top:2px;">${new Date().toLocaleTimeString()}</div>
      </div>`;
    silentPrint(labelHtml, '101.6mm', '50.8mm').then(result => {
      console.log('[BL] Print result:', result);
      sendResponse(result);
    });
    return true;
  }

  if (msg.type === 'clear_sales') {
    totalSales = 0;
    chrome.action.setBadgeText({ text: '' });
    sendResponse({ success: true });
    return true;
  }
});

// Set defaults on install
chrome.runtime.onInstalled.addListener(() => {
  chrome.storage.local.get(['serverUrl'], (result) => {
    if (!result.serverUrl) {
      chrome.storage.local.set({ serverUrl: 'https://bundlelive.com' });
    }
  });
  chrome.action.setBadgeText({ text: '' });
  // Immediate checks on install
  checkForUpdate();
  fetchRemoteConfigs();
});
