/**
 * BundleLive Bookmarklet v2
 * 
 * Captures sales from Whatnot live shows via:
 * 1. DOM mutation observer (watches for "username won!" text)
 * 2. WebSocket interception (new connections)
 * 3. Fetch/GraphQL interception
 * 4. Periodic DOM scanning (catches what MutationObserver misses)
 */

(function(){
  // If already running, don't double-load
  if (window.__bundlelive_active) { alert('BundleLive is already running!'); return; }
  window.__bundlelive_active = true;

  var SERVER = prompt('Enter your BundleLive server URL:', 'http://localhost:3456');
  if (!SERVER) { window.__bundlelive_active = false; return; }
  SERVER = SERVER.replace(/\/$/,'');
  
  var seen = {};
  var count = 0;
  
  // Floating status badge
  var badge = document.createElement('div');
  badge.id = 'bl-badge';
  badge.innerHTML = '⚡ BundleLive: Connecting...';
  badge.style.cssText = 'position:fixed;top:10px;right:10px;z-index:99999;background:#0d0d1a;color:#fbbf24;padding:10px 18px;border-radius:20px;font-family:system-ui,sans-serif;font-size:14px;font-weight:700;border:2px solid #fbbf24;cursor:pointer;box-shadow:0 4px 20px rgba(0,0,0,0.5);transition:all 0.3s;';
  document.body.appendChild(badge);
  
  function updateBadge() {
    badge.innerHTML = '⚡ BundleLive: ' + count + ' sale' + (count !== 1 ? 's' : '');
    badge.style.borderColor = '#4ade80';
    badge.style.color = '#4ade80';
  }

  function showError(msg) {
    badge.style.borderColor = '#ef4444';
    badge.style.color = '#ef4444';
    badge.innerHTML = '⚡ BundleLive: ' + msg;
  }

  // Test server connection first
  fetch(SERVER + '/api/shows/active')
    .then(function(r){ return r.json(); })
    .then(function(d){
      if (d.show) {
        badge.innerHTML = '⚡ BundleLive: 0 sales';
        badge.style.borderColor = '#4ade80';
        badge.style.color = '#4ade80';
        console.log('[BundleLive] Connected! Active show: ' + d.show.title);
      } else {
        showError('No active show!');
        console.log('[BundleLive] WARNING: No active show on server. Start a show first.');
      }
    })
    .catch(function(e){
      showError('Server offline');
      console.error('[BundleLive] Cannot reach server at ' + SERVER, e);
    });

  function send(sale) {
    // Dedup by multiple keys
    var keys = [sale.id, sale.buyer + '_' + sale.price + '_' + Math.floor(Date.now()/3000)];
    for (var i = 0; i < keys.length; i++) {
      if (seen[keys[i]]) return;
    }
    keys.forEach(function(k){ seen[k] = true; });
    setTimeout(function(){ keys.forEach(function(k){ delete seen[k]; }); }, 10000);
    
    count++;
    updateBadge();
    console.log('[BundleLive] Sale #' + count + ': @' + sale.buyer + ' — ' + sale.item + ' $' + sale.price);
    
    fetch(SERVER + '/api/sales/from-extension', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({
        event_id: sale.id,
        buyer_username: sale.buyer,
        item_name: sale.item || 'Live Sale',
        price: sale.price,
        source: 'bookmarklet',
        timestamp: new Date().toISOString()
      })
    }).then(function(r){ return r.json(); }).then(function(d){
      if (d.success) {
        console.log('[BundleLive] → Bin ' + (d.bin && d.bin.label || '#' + d.bin.bin_number));
        // Flash badge
        badge.style.background = '#14532d';
        setTimeout(function(){ badge.style.background = '#0d0d1a'; }, 300);
      } else {
        console.warn('[BundleLive] Server rejected:', d.error);
      }
    }).catch(function(e){
      showError('Send failed');
      console.error('[BundleLive] Send error:', e);
    });
  }

  // ─── METHOD 1: DOM Mutation Observer ────────────────────
  // Whatnot shows sales as: "username won!" in the chat/feed
  var domPatterns = [
    /(\w{2,30})\s+won!/i,                          // "username won!"
    /sold\s+to\s+@?(\w{2,30})/i,                   // "Sold to @username"
    /winner:\s*@?(\w{2,30})/i,                       // "Winner: username"
    /congratulations?\s+@?(\w{2,30})/i,              // "Congrats username"
    /(\w{2,30})\s+(?:bought|purchased)/i             // "username bought"
  ];

  function extractSaleFromText(text) {
    for (var i = 0; i < domPatterns.length; i++) {
      var match = text.match(domPatterns[i]);
      if (match) {
        var buyer = match[1];
        // Skip common false positives
        if (/^(joined|left|says|typed|the|and|you|has|have|was|get|for)$/i.test(buyer)) continue;
        var priceMatch = text.match(/\$(\d+(?:\.\d{2})?)/);
        return {
          buyer: buyer,
          price: priceMatch ? parseFloat(priceMatch[1]) : 0,
          item: text.substring(0, 80).trim()
        };
      }
    }
    return null;
  }

  // Watch for new DOM nodes
  var observer = new MutationObserver(function(muts) {
    muts.forEach(function(m) {
      m.addedNodes.forEach(function(n) {
        if (n.nodeType !== 1) return;
        var text = n.textContent || '';
        if (text.length < 5 || text.length > 500) return;
        
        // Check for "won!" pattern
        if (text.indexOf('won!') > -1 || text.indexOf('sold') > -1 || text.indexOf('Sold') > -1 || text.indexOf('winner') > -1) {
          var sale = extractSaleFromText(text);
          if (sale) {
            send({
              id: 'dom_' + Date.now() + '_' + sale.buyer,
              buyer: sale.buyer,
              item: sale.item,
              price: sale.price
            });
          }
        }
      });
    });
  });
  observer.observe(document.body, { childList: true, subtree: true });

  // ─── METHOD 2: Periodic DOM scan ───────────────────────
  // Fallback: scan for sale indicators every 2 seconds
  var lastScanText = '';
  setInterval(function() {
    // Look for the sale result area (usually shows "username won!" near the item)
    var elements = document.querySelectorAll('[class*="sold"], [class*="winner"], [class*="won"], [data-testid*="sale"], [data-testid*="winner"]');
    elements.forEach(function(el) {
      var text = el.textContent || '';
      if (text === lastScanText) return;
      if (text.indexOf('won') > -1) {
        var sale = extractSaleFromText(text);
        if (sale) {
          lastScanText = text;
          send({
            id: 'scan_' + Date.now() + '_' + sale.buyer,
            buyer: sale.buyer,
            item: sale.item,
            price: sale.price
          });
        }
      }
    });
    
    // Also scan the visible "Sold" text near the price
    var allText = document.body.innerText;
    var wonMatches = allText.match(/(\w{2,30})\s+won!/gi);
    if (wonMatches) {
      var latest = wonMatches[wonMatches.length - 1];
      if (latest !== lastScanText) {
        var sale = extractSaleFromText(latest);
        if (sale) {
          lastScanText = latest;
          // Try to find price nearby
          var priceArea = document.querySelector('[class*="price"], [class*="amount"], [class*="bid"]');
          if (priceArea && !sale.price) {
            var pm = priceArea.textContent.match(/\$(\d+(?:\.\d{2})?)/);
            if (pm) sale.price = parseFloat(pm[1]);
          }
          send({
            id: 'scan2_' + Date.now() + '_' + sale.buyer,
            buyer: sale.buyer,
            item: sale.item,
            price: sale.price
          });
        }
      }
    }
  }, 2000);

  // ─── METHOD 3: WebSocket interception ──────────────────
  var OrigWS = window.WebSocket;
  window.WebSocket = function(url, proto) {
    console.log('[BundleLive] New WebSocket:', url);
    var ws = proto ? new OrigWS(url, proto) : new OrigWS(url);
    ws.addEventListener('message', function(e) {
      try {
        var d = JSON.parse(e.data);
        var s = JSON.stringify(d).toLowerCase();
        if (s.indexOf('sold') > -1 || s.indexOf('purchase') > -1 || s.indexOf('bid_won') > -1 || s.indexOf('won') > -1) {
          var buyer = (d.buyer && d.buyer.username) || (d.winner && d.winner.username) || d.username || d.buyerUsername;
          if (buyer) {
            send({
              id: d.id || 'ws_' + Date.now(),
              buyer: buyer,
              item: (d.listing && d.listing.title) || d.title || d.itemName || 'Live Sale',
              price: (d.priceCents || d.price || 0) / 100
            });
          }
        }
      } catch(ex) {}
    });
    return ws;
  };
  window.WebSocket.prototype = OrigWS.prototype;
  window.WebSocket.CONNECTING = OrigWS.CONNECTING;
  window.WebSocket.OPEN = OrigWS.OPEN;
  window.WebSocket.CLOSING = OrigWS.CLOSING;
  window.WebSocket.CLOSED = OrigWS.CLOSED;

  // ─── METHOD 4: Fetch/GraphQL interception ──────────────
  var origFetch = window.fetch;
  window.fetch = function() {
    var args = arguments;
    return origFetch.apply(this, args).then(function(resp) {
      try {
        var url = typeof args[0] === 'string' ? args[0] : (args[0] && args[0].url || '');
        if (url.indexOf('graphql') > -1 || url.indexOf('whatnot') > -1) {
          resp.clone().json().then(function(data) {
            var s = JSON.stringify(data).toLowerCase();
            if (s.indexOf('buyer') > -1 && (s.indexOf('price') > -1 || s.indexOf('cents') > -1 || s.indexOf('won') > -1)) {
              JSON.stringify(data, function(k, v) {
                if (v && typeof v === 'object' && v.buyer && (v.priceCents || v.totalCents || v.price)) {
                  var buyerName = (typeof v.buyer === 'object') ? v.buyer.username : v.buyer;
                  if (buyerName) {
                    send({
                      id: v.id || 'gql_' + Date.now(),
                      buyer: buyerName,
                      item: (v.listing && v.listing.title) || v.title || 'Live Sale',
                      price: (v.priceCents || v.totalCents || 0) / 100
                    });
                  }
                }
                return v;
              });
            }
          }).catch(function(){});
        }
      } catch(ex) {}
      return resp;
    });
  };

  console.log('[BundleLive] Bookmarklet v2 loaded — watching for sales via DOM + WebSocket + Fetch + Scan');
  alert('⚡ BundleLive connected!\n\nWatching for sales on this page.\nKeep this tab open during your show.');
})();
