/**
 * BundleLive Capture v3.8.0 - Headless capture engine
 * Auto-runs on whatnot.com/live/* pages
 *
 * v3.7.2 FIXES:
 * - Dedup uses buyer+price+timestamp_bucket (NOT item name — getItem() is unreliable)
 * - 2 second dedup window (fast auctions)
 * - DOM observer also watches characterData + attribute changes (Whatnot updates existing nodes)
 * - Scan checks ALL "won!" matches, not just the last one
 * - Smart Chat: added debug logging + retry on failure
 * - Smart Chat fires BEFORE server sale POST (don't wait for server)
 * - getPrice() improved to grab price near "won!" text
 */
(function(){
  if (window.__bundlelive_active) return;
  window.__bundlelive_active = true;

  let SERVER = 'https://bundlelive.com';
  let AUTH_TOKEN = '';
  let showRunning = false;
  let captureEnabled = true;
  let userStopped = false;  // TRUE when user clicks End Show — prevents auto-restart
  let smartChatOn = false;
  let autoPrintOn = false;
  let salesCount = 0;
  let totalRevenue = 0;
  const uniqueBuyers = new Set();
  const seen = {};
  const sentSales = {};
  const smartChatSent = {};

  // Listen for clear command from popup
  if (typeof chrome !== 'undefined' && chrome.runtime) {
    chrome.runtime.onMessage.addListener(function(msg) {
      if (msg.type === 'clear_all_state') {
        salesCount = 0;
        totalRevenue = 0;
        uniqueBuyers.clear();
        for (var k in sentSales) delete sentSales[k];
        for (var s in seen) delete seen[s];
        for (var c in smartChatSent) delete smartChatSent[c];
        lastSeenWinners = {};
        if (typeof chrome !== 'undefined' && chrome.storage) {
          chrome.storage.local.set({
            salesCount: 0, totalRevenue: 0, uniqueBuyers: 0,
            uniqueBuyersList: [], sentSaleKeys: {}
          });
        }
        console.log('[BL] ✅ All state cleared');
      }
      // User stopped the show — stop capturing until explicitly restarted
      if (msg.type === 'stop_capture') {
        userStopped = true;
        showRunning = false;
        captureEnabled = false;
        if (typeof chrome !== 'undefined' && chrome.storage) {
          chrome.storage.local.set({ userStopped: true });
        }
        console.log('[BL] 🛑 User stopped capture — will not auto-restart');
      }
      // User explicitly started a show — resume capturing
      if (msg.type === 'start_capture') {
        userStopped = false;
        captureEnabled = true;
        if (typeof chrome !== 'undefined' && chrome.storage) {
          chrome.storage.local.set({ userStopped: false });
        }
        console.log('[BL] ▶️ User started capture');
      }
    });
  }

  // Restore state from storage
  if (typeof chrome !== 'undefined' && chrome.storage) {
    chrome.storage.local.get(['salesCount', 'totalRevenue', 'uniqueBuyersList', 'userStopped'], function(r) {
      if (r.salesCount) salesCount = r.salesCount;
      if (r.totalRevenue) totalRevenue = r.totalRevenue;
      if (r.uniqueBuyersList && Array.isArray(r.uniqueBuyersList)) {
        r.uniqueBuyersList.forEach(function(b) { uniqueBuyers.add(b); });
      }
      if (r.userStopped) {
        userStopped = true;
        captureEnabled = false;
        console.log('[BL] 🛑 Restored userStopped state — capture disabled until user starts');
      }
    });
  }

  // Cleanup stale dedup keys every 20s
  setInterval(function() {
    var now = Date.now();
    for (var key in seen) {
      if (now - seen[key] > 10000) delete seen[key];
    }
    for (var sk in sentSales) {
      if (now - sentSales[sk] > 10000) delete sentSales[sk];
    }
    for (var sc in smartChatSent) {
      if (now - smartChatSent[sc] > 8000) delete smartChatSent[sc];
    }
  }, 20000);

  // Load settings
  if (typeof chrome !== 'undefined' && chrome.storage) {
    chrome.storage.local.get(['serverUrl', 'smartChatEnabled', 'autoPrintEnabled', 'authToken'], function(r) {
      if (r.serverUrl) SERVER = r.serverUrl;
      if (r.authToken) AUTH_TOKEN = r.authToken;
      smartChatOn = !!r.smartChatEnabled;
      autoPrintOn = !!r.autoPrintEnabled;
      console.log('[BL] Settings: SmartChat=' + smartChatOn + ' AutoPrint=' + autoPrintOn + ' Auth=' + !!AUTH_TOKEN);
      if (AUTH_TOKEN) syncWithServer();
    });
    chrome.storage.onChanged.addListener(function(changes) {
      if (changes.serverUrl) SERVER = changes.serverUrl.newValue;
      if (changes.smartChatEnabled) { smartChatOn = changes.smartChatEnabled.newValue; console.log('[BL] SmartChat toggled:', smartChatOn); }
      if (changes.autoPrintEnabled) { autoPrintOn = changes.autoPrintEnabled.newValue; }
      if (changes.authToken && changes.authToken.newValue) {
        AUTH_TOKEN = changes.authToken.newValue;
        syncWithServer();
      }
    });
  }

  function authHeaders() {
    var h = { 'Content-Type': 'application/json' };
    if (AUTH_TOKEN) h['Authorization'] = 'Bearer ' + AUTH_TOKEN;
    return h;
  }

  if (typeof chrome !== 'undefined' && chrome.runtime) {
    chrome.runtime.sendMessage({ type: 'get_auth_token' }, function(response) {
      if (response && response.token && !AUTH_TOKEN) {
        AUTH_TOKEN = response.token;
        syncWithServer();
      }
    });
  }

  // ─── SERVER SYNC ─────────────────────────────────────
  function syncWithServer() {
    if (!AUTH_TOKEN) return;
    fetch(SERVER + '/api/shows/active', { headers: authHeaders() })
      .then(function(r) { return r.json(); })
      .then(function(d) {
        if (d.success && d.show) {
          // Only enable capture if user hasn't manually stopped
          if (!userStopped) {
            showRunning = true;
            captureEnabled = true;
          }
          if (typeof chrome !== 'undefined' && chrome.runtime) {
            try { chrome.runtime.sendMessage({ type: 'connected', showActive: true }); } catch(e) {}
          }
        } else {
          showRunning = false;
          // If server has no show, also clear userStopped so next show starts fresh
          if (userStopped) {
            userStopped = false;
            if (typeof chrome !== 'undefined' && chrome.storage) {
              chrome.storage.local.set({ userStopped: false });
            }
          }
        }
      })
      .catch(function(e) { console.log('[BL] Sync error:', e.message); });
  }
  setInterval(function() { if (AUTH_TOKEN) syncWithServer(); }, 5000);

  // ─── PRICE SCRAPING ─────────────────────────────────
  // NOTE: DOM price scraping is a fallback. WS and GQL methods provide structured price data.
  function getPrice() {
    try {
      // Strategy 1: Find the current bid/sold price element on the Whatnot stream page
      // Whatnot shows the current price prominently — look for specific containers first
      var priceContainers = document.querySelectorAll('[data-testid*="price" i], [class*="currentBid" i], [class*="current-bid" i], [class*="sold-price" i], [class*="winning-bid" i]');
      for (var pc = 0; pc < priceContainers.length; pc++) {
        var pcMatch = (priceContainers[pc].textContent || '').match(/\$(\d+(?:,\d{3})*(?:\.\d{2})?)/);
        if (pcMatch) return parseFloat(pcMatch[1].replace(',', ''));
      }

      // Strategy 2: Find price near the most recent "won!" or "Sold" text
      var all = document.body.innerText;
      var wonIdx = all.lastIndexOf('won!');
      var soldIdx = all.lastIndexOf('Sold');
      var idx = Math.max(wonIdx, soldIdx);
      if (idx > -1) {
        // Look in a tight window around the sale event
        var start = Math.max(0, idx - 150);
        var end = Math.min(all.length, idx + 100);
        var chunk = all.substring(start, end);
        // Match dollar amounts — prefer ones right before "won!" (the bid amount)
        var beforeWon = all.substring(Math.max(0, idx - 100), idx);
        var beforeMatches = beforeWon.match(/\$(\d+(?:,\d{3})*(?:\.\d{2})?)/g);
        if (beforeMatches) {
          var lastMatch = beforeMatches[beforeMatches.length - 1];
          return parseFloat(lastMatch.replace('$', '').replace(',', ''));
        }
        // Fallback: after the sale text
        var afterMatches = chunk.match(/\$(\d+(?:,\d{3})*(?:\.\d{2})?)/g);
        if (afterMatches) {
          return parseFloat(afterMatches[0].replace('$', '').replace(',', ''));
        }
      }

      // Strategy 3: Look for price elements by class
      var priceEls = document.querySelectorAll('[class*="price" i], [class*="amount" i], [class*="cost" i]');
      for (var p = 0; p < priceEls.length; p++) {
        var m = (priceEls[p].textContent || '').match(/\$(\d+(?:,\d{3})*(?:\.\d{2})?)/);
        if (m) {
          var val = parseFloat(m[1].replace(',', ''));
          if (val > 0.5 && val < 50000) return val;
        }
      }
    } catch(e) {}
    return 0;
  }

  function getItem() {
    try {
      // Strategy 1: Find the listing title element on Whatnot's stream page
      // Whatnot typically has a title/listing element showing the current item
      var titleEls = document.querySelectorAll('[data-testid*="title" i], [data-testid*="listing" i], [class*="listingTitle" i], [class*="listing-title" i], [class*="itemTitle" i], [class*="item-title" i]');
      for (var j = 0; j < titleEls.length; j++) {
        var text = (titleEls[j].textContent || '').trim();
        if (text.length > 3 && text.length < 200) {
          return text.substring(0, 200);  // Keep FULL name including #number
        }
      }

      // Strategy 2: Look for item name near "won!" text
      var all = document.body.innerText;
      var wonIdx = all.lastIndexOf('won!');
      if (wonIdx > -1) {
        // Check BEFORE "won!" — the item name is usually above/before the sale notification
        var beforeChunk = all.substring(Math.max(0, wonIdx - 500), wonIdx);
        var beforeLines = beforeChunk.split('\n');
        // Walk backwards to find item name (usually has # in it for lot number)
        for (var i = beforeLines.length - 1; i >= 0; i--) {
          var t = beforeLines[i].trim();
          if (t.length > 5 && t.length < 200 && !t.match(/^(won!|Sold|Bid|Shipping|Awaiting|say something|\$)/i)) {
            if (t.match(/#\d+/) || t.match(/lot|pack|card|coin|shoe|item|random|pull/i)) {
              return t.substring(0, 200);  // Keep FULL name — do NOT strip $ or truncate
            }
          }
        }
        // Also check after "won!"
        var afterChunk = all.substring(wonIdx, Math.min(all.length, wonIdx + 300));
        var afterLines = afterChunk.split('\n');
        for (var a = 0; a < afterLines.length; a++) {
          var at = afterLines[a].trim();
          if (at.length > 5 && at.length < 200 && !at.match(/^(won!|Sold|Bid|Shipping|Awaiting|say something|\$)/i)) {
            if (at.match(/#\d+/) || at.match(/lot|pack|card|coin|shoe|item|random|pull/i)) {
              return at.substring(0, 200);
            }
          }
        }
      }

      // Strategy 3: Find any element with title-like class containing #number
      var allTitleEls = document.querySelectorAll('[class*="title" i], [class*="item" i], [class*="listing" i]');
      for (var k = 0; k < allTitleEls.length; k++) {
        var elText = (allTitleEls[k].textContent || '').trim();
        if (elText.length > 5 && elText.length < 200 && elText.match(/#\d+/)) {
          return elText.substring(0, 200);
        }
      }
    } catch(e) {}
    return 'Live Sale';
  }

  // ─── SEND SALE ──────────────────────────────────────
  // saleCounter ensures each sale gets a unique key even with same buyer+price
  var saleCounter = 0;

  function sendSale(buyer, price, item, source) {
    if (!AUTH_TOKEN) { console.log('[BL] ⚠️ No auth'); return; }
    if (userStopped || !captureEnabled) { console.log('[BL] ⏸️ Capture stopped by user'); return; }
    buyer = (buyer || '').toLowerCase().trim();
    if (!buyer || buyer.length < 2) return;

    var now = Date.now();

    // ─── DEDUP: 2 second window using buyer + price + time bucket ───
    // Each 2-second window allows ONE sale per buyer+price combo
    // This means same buyer winning at same price 3 seconds later goes through
    var timeBucket = Math.floor(now / 2000);
    var normalizedKey = buyer + '|' + (parseFloat(price) || 0).toFixed(2) + '|' + timeBucket;
    if (sentSales[normalizedKey]) {
      console.log('[BL] ⏭️ Dedup (' + source + '): ' + buyer + ' $' + price);
      return;
    }
    sentSales[normalizedKey] = now;

    saleCounter++;
    salesCount++;
    totalRevenue += (price || 0);
    uniqueBuyers.add(buyer);

    if (typeof chrome !== 'undefined' && chrome.storage) {
      chrome.storage.local.set({
        salesCount: salesCount, totalRevenue: totalRevenue,
        uniqueBuyers: uniqueBuyers.size, uniqueBuyersList: Array.from(uniqueBuyers)
      });
    }

    if (typeof chrome !== 'undefined' && chrome.runtime) {
      try { chrome.runtime.sendMessage({ type: 'sale_captured', count: salesCount }); } catch(e) {}
    }

    console.log('[BL] 🎯 Sale #' + salesCount + ': @' + buyer + ' - ' + item + ' $' + price + ' (' + source + ')');

    // ─── Send sale to server (Smart Chat fires AFTER sale is saved) ───
    var screenshot = '';
    try {
      var video = document.querySelector('video');
      if (video && !video.paused && video.videoWidth > 0) {
        var c = document.createElement('canvas');
        var s = Math.min(1, 320 / video.videoWidth);
        c.width = Math.round(video.videoWidth * s);
        c.height = Math.round(video.videoHeight * s);
        c.getContext('2d').drawImage(video, 0, 0, c.width, c.height);
        screenshot = c.toDataURL('image/jpeg', 0.4);
      }
    } catch(e) {}

    var body = {
      event_id: 'sale_' + saleCounter + '_' + buyer + '_' + now,
      buyer_username: buyer,
      item_name: item || 'Live Sale',
      price: price,
      screenshot: screenshot,
      source: 'extension_' + source,
      timestamp: new Date().toISOString()
    };

    fetch(SERVER + '/api/sales/from-extension', {
      method: 'POST',
      headers: authHeaders(),
      body: JSON.stringify(body)
    })
    .then(function(r) { return r.json(); })
    .then(function(d) {
      if (d.success) {
        if (!showRunning) { showRunning = true; }
        if (d.bin) console.log('[BL] 📦 Bin ' + (d.bin.label || '#' + d.bin.bin_number));

        // ─── SMART CHAT — fires AFTER sale saved, ONE per buyer per 5 seconds ───
        if (smartChatOn) {
          var scNow = Date.now();
          var scKey = buyer;
          if (!smartChatSent[scKey] || (scNow - smartChatSent[scKey]) > 5000) {
            smartChatSent[scKey] = scNow;
            console.log('[BL] 💬 SmartChat: requesting message for @' + buyer);
            fetch(SERVER + '/api/smart-chat/send', {
              method: 'POST',
              headers: authHeaders(),
              body: JSON.stringify({ buyer_username: buyer, item_name: item, price: price })
            })
            .then(function(r) { return r.json(); })
            .then(function(chatData) {
              if (chatData.success && chatData.message) {
                console.log('[BL] 💬 SmartChat: "' + chatData.message + '"');
                setTimeout(function() { typeIntoWhatnotChat(chatData.message); }, 400);
              } else {
                console.log('[BL] 💬 SmartChat skipped:', chatData.reason || 'no message');
              }
            })
            .catch(function(e) { console.log('[BL] 💬 SmartChat error:', e.message); });
          }
        }
      } else {
        console.log('[BL] ⚠️ Server:', JSON.stringify(d));
      }
    })
    .catch(function(err) {
      console.log('[BL] ⚠️ Retrying sale...', err.message);
      setTimeout(function() {
        delete body.screenshot;
        fetch(SERVER + '/api/sales/from-extension', {
          method: 'POST', headers: authHeaders(), body: JSON.stringify(body)
        }).catch(function() {});
      }, 2000);
    });
  }

  // ═══════════════════════════════════════════════════════
  // SMART CHAT — Type into Whatnot chat
  // ═══════════════════════════════════════════════════════
  function typeIntoWhatnotChat(message) {
    try {
      var input = null;

      // Strategy 1: "Say something..." placeholder (EXACT match for Whatnot)
      var allInputs = document.querySelectorAll('input, textarea');
      for (var i = 0; i < allInputs.length; i++) {
        var ph = (allInputs[i].placeholder || '').toLowerCase();
        if (ph.indexOf('say something') > -1) {
          input = allInputs[i];
          console.log('[BL] 💬 ✓ Found "Say something" input');
          break;
        }
      }

      // Strategy 2: broader placeholder match
      if (!input) {
        for (var i2 = 0; i2 < allInputs.length; i2++) {
          var ph2 = (allInputs[i2].placeholder || '').toLowerCase();
          if (ph2.indexOf('chat') > -1 || ph2.indexOf('message') > -1 || ph2.indexOf('comment') > -1 || ph2.indexOf('type') > -1) {
            input = allInputs[i2];
            console.log('[BL] 💬 ✓ Found input via placeholder:', ph2);
            break;
          }
        }
      }

      // Strategy 3: aria-label
      if (!input) {
        input = document.querySelector('[aria-label*="chat" i], [aria-label*="message" i], [aria-label*="say something" i]');
        if (input) console.log('[BL] 💬 ✓ Found via aria-label');
      }

      // Strategy 4: role=textbox or data-testid
      if (!input) {
        input = document.querySelector('[role="textbox"], [data-testid*="chat" i], [data-testid*="message" i]');
        if (input) console.log('[BL] 💬 ✓ Found via role/testid');
      }

      // Strategy 5: contenteditable
      if (!input) {
        var editables = document.querySelectorAll('[contenteditable="true"]');
        for (var e = 0; e < editables.length; e++) {
          if (editables[e].offsetHeight > 0 && editables[e].offsetHeight < 200 && editables[e].offsetWidth > 50) {
            input = editables[e];
            console.log('[BL] 💬 ✓ Found via contenteditable');
            break;
          }
        }
      }

      // Strategy 6: walk up from "Say something" text
      if (!input) {
        var spans = document.querySelectorAll('span, div, p, label');
        for (var j = 0; j < spans.length; j++) {
          var txt = (spans[j].textContent || '').trim().toLowerCase();
          if (txt === 'say something...' || txt === 'say something') {
            var parent = spans[j].parentElement;
            for (var k = 0; k < 10 && parent; k++) {
              var found = parent.querySelector('input, textarea, [contenteditable="true"], [role="textbox"]');
              if (found) { input = found; console.log('[BL] 💬 ✓ Found via text walk'); break; }
              parent = parent.parentElement;
            }
            if (input) break;
          }
        }
      }

      // Strategy 7: any visible text input
      if (!input) {
        var vis = document.querySelectorAll('input[type="text"], input:not([type]), textarea');
        for (var v = 0; v < vis.length; v++) {
          if (vis[v].offsetHeight > 0 && vis[v].offsetWidth > 50 && !(vis[v].placeholder || '').match(/search/i)) {
            input = vis[v];
            console.log('[BL] 💬 ✓ Found via visible scan');
            break;
          }
        }
      }

      if (!input) {
        console.log('[BL] 💬 ❌ NO CHAT INPUT FOUND. Inputs on page:', document.querySelectorAll('input, textarea').length);
        // Log what we can see
        var allI = document.querySelectorAll('input, textarea');
        for (var d = 0; d < allI.length; d++) {
          console.log('[BL] 💬   input:', allI[d].tagName, 'type:', allI[d].type, 'ph:', (allI[d].placeholder || '').substring(0, 40), 'visible:', allI[d].offsetHeight > 0);
        }
        return;
      }

      var isContentEditable = input.tagName !== 'INPUT' && input.tagName !== 'TEXTAREA';
      console.log('[BL] 💬 Using input:', input.tagName, 'contentEditable:', isContentEditable, 'placeholder:', (input.placeholder || '').substring(0, 30));

      // ── Focus and type ──
      input.focus();
      input.click();

      setTimeout(function() {
        if (isContentEditable) {
          input.textContent = '';
          input.focus();
          input.textContent = message;
          var ceTracker = input._valueTracker;
          if (ceTracker) ceTracker.setValue('');
          input.dispatchEvent(new InputEvent('input', { bubbles: true, inputType: 'insertText', data: message }));
          input.dispatchEvent(new Event('change', { bubbles: true }));
        } else {
          // React native value setter trick
          var proto = input.tagName === 'TEXTAREA' ? window.HTMLTextAreaElement.prototype : window.HTMLInputElement.prototype;
          var setter = Object.getOwnPropertyDescriptor(proto, 'value');
          if (setter && setter.set) setter.set.call(input, message);
          else input.value = message;
          var tracker = input._valueTracker;
          if (tracker) tracker.setValue('');
          input.dispatchEvent(new InputEvent('input', { bubbles: true, inputType: 'insertText', data: message }));
          input.dispatchEvent(new Event('change', { bubbles: true }));
          console.log('[BL] 💬 Value set to:', input.value.substring(0, 40));
        }

        // ── Send via Enter key ──
        setTimeout(function() {
          input.dispatchEvent(new KeyboardEvent('keydown', {
            key: 'Enter', code: 'Enter', keyCode: 13, which: 13,
            bubbles: true, cancelable: true
          }));
          console.log('[BL] 💬 Enter dispatched');

          // Verify + fallback to button click
          setTimeout(function() {
            var val = isContentEditable ? (input.textContent || '').trim() : (input.value || '').trim();
            if (val.length === 0 || val !== message) {
              console.log('[BL] 💬 ✅ Message sent via Enter');
              return;
            }
            console.log('[BL] 💬 Enter didn\'t clear input, trying button...');

            // Find send button
            var sendBtn = document.querySelector('[aria-label*="send" i], [aria-label*="Send"], button[type="submit"], [data-testid*="send" i]');
            if (!sendBtn) {
              var buttons = document.querySelectorAll('button');
              for (var b = 0; b < buttons.length; b++) {
                if (buttons[b].offsetHeight > 0 && buttons[b].offsetHeight < 60) {
                  var btnRect = buttons[b].getBoundingClientRect();
                  var inputRect = input.getBoundingClientRect();
                  if (Math.abs(btnRect.top - inputRect.top) < 60 && btnRect.left > inputRect.left) {
                    sendBtn = buttons[b];
                    break;
                  }
                }
              }
            }
            if (sendBtn) {
              sendBtn.click();
              console.log('[BL] 💬 ✅ Message sent via button click');
            } else {
              console.log('[BL] 💬 ⚠️ No send button found');
            }
          }, 250);
        }, 300);
      }, 100);

    } catch(err) {
      console.log('[BL] 💬 ERROR:', err.message);
    }
  }

  // ─── METHOD 1: DOM Mutation Observer ────────────────
  // Watch for BOTH new nodes AND text changes (Whatnot may update existing nodes)
  var domPatterns = [
    /(\w{2,30})\s+won!/i,
    /sold\s+to\s+@?(\w{2,30})/i,
    /winner[:\s]+@?(\w{2,30})/i,
    /congratulations?\s+@?(\w{2,30})/i,
    /(\w{2,30})\s+(?:bought|purchased|claimed)/i,
    /(\w{2,30})\s+got\s+it/i
  ];
  var skipBuyers = /^(joined|left|the|and|you|has|get|for|say|is|was|are|been|this|that|just|not|will|can|all|now|it|buy|bid|sold|won|live|show|new|hot|price|item|lot|pre)$/i;

  function checkTextForSale(text) {
    if (!AUTH_TOKEN) return;
    if (!text || text.length < 3 || text.length > 500) return;
    var lowerText = text.toLowerCase();
    if (lowerText.indexOf('won') === -1 && lowerText.indexOf('sold') === -1 && lowerText.indexOf('winner') === -1 && lowerText.indexOf('bought') === -1 && lowerText.indexOf('claimed') === -1) return;

    for (var pi = 0; pi < domPatterns.length; pi++) {
      var match = text.match(domPatterns[pi]);
      if (match) {
        var buyer = match[1];
        if (skipBuyers.test(buyer)) return;
        var key = 'dom_' + text.substring(0, 50);
        if (seen[key] && (Date.now() - seen[key]) < 2000) return;
        seen[key] = Date.now();
        sendSale(buyer, getPrice(), getItem(), 'dom');
        return;
      }
    }
  }

  var observer = new MutationObserver(function(muts) {
    for (var mi = 0; mi < muts.length; mi++) {
      var mut = muts[mi];

      // New nodes added
      if (mut.addedNodes) {
        for (var ni = 0; ni < mut.addedNodes.length; ni++) {
          var n = mut.addedNodes[ni];
          if (n.nodeType === 1) checkTextForSale((n.textContent || '').trim());
          if (n.nodeType === 3) checkTextForSale((n.nodeValue || '').trim());
        }
      }

      // Text content changed in existing nodes
      if (mut.type === 'characterData') {
        checkTextForSale((mut.target.nodeValue || '').trim());
      }
    }
  });
  observer.observe(document.body, {
    childList: true,
    subtree: true,
    characterData: true // CATCH TEXT CHANGES in existing nodes
  });

  // ─── METHOD 2: Periodic Scan (every 700ms) ─────────
  // Track ALL seen winners, not just the last one
  var lastSeenWinners = {}; // "buyer won!" text → timestamp

  setInterval(function() {
    if (!AUTH_TOKEN) return;
    var text = document.body.innerText;
    var matches = text.match(/(\w{2,30})\s+won!/gi);
    if (!matches) return;

    var now = Date.now();
    for (var i = 0; i < matches.length; i++) {
      var m = matches[i];
      // Skip if we've seen this exact text recently
      if (lastSeenWinners[m] && (now - lastSeenWinners[m]) < 2000) continue;

      var parsed = m.match(/(\w{2,30})\s+won!/i);
      if (parsed && !skipBuyers.test(parsed[1])) {
        lastSeenWinners[m] = now;
        var scanKey = 'scan_' + m;
        if (!seen[scanKey] || (now - seen[scanKey]) >= 2000) {
          seen[scanKey] = now;
          sendSale(parsed[1], getPrice(), getItem(), 'scan');
        }
      }
    }

    // Clean old entries
    for (var key in lastSeenWinners) {
      if (now - lastSeenWinners[key] > 10000) delete lastSeenWinners[key];
    }
  }, 700);

  // ─── METHOD 3: WebSocket Interception ───────────────
  var OrigWS = window.WebSocket;
  window.WebSocket = function(url, proto) {
    var ws = proto ? new OrigWS(url, proto) : new OrigWS(url);
    ws.addEventListener('message', function(e) {
      if (!AUTH_TOKEN) return;
      try {
        var d = JSON.parse(e.data);
        var s = JSON.stringify(d).toLowerCase();
        if (s.indexOf('sold') > -1 || s.indexOf('purchase') > -1 || s.indexOf('bid_won') > -1 || s.indexOf('won') > -1 || s.indexOf('claimed') > -1) {
          var buyer = (d.buyer && d.buyer.username) || (d.winner && d.winner.username) || d.username || d.buyerUsername || d.buyer_username;
          if (buyer && !skipBuyers.test(buyer)) {
            var price = d.priceCents ? d.priceCents / 100 : (d.totalCents ? d.totalCents / 100 : (d.price || 0));
            var item = (d.listing && d.listing.title) || d.title || d.itemName || d.item_name || 'Live Sale';
            sendSale(buyer, price, item, 'ws');
          }
        }
      } catch(ex) {}
    });
    return ws;
  };
  window.WebSocket.prototype = OrigWS.prototype;
  window.WebSocket.CONNECTING = OrigWS.CONNECTING;
  window.WebSocket.OPEN = OrigWS.OPEN;
  window.WebSocket.CLOSING = OrigWS.CLOSING;
  window.WebSocket.CLOSED = OrigWS.CLOSED;

  // ─── METHOD 4: Fetch/GraphQL Interception ───────────
  var origFetch = window.fetch;
  window.fetch = function() {
    var args = arguments;
    return origFetch.apply(this, args).then(function(resp) {
      if (!AUTH_TOKEN) return resp;
      try {
        var url = typeof args[0] === 'string' ? args[0] : (args[0] && args[0].url || '');
        if ((url.indexOf('graphql') > -1 || url.indexOf('whatnot') > -1) && url.indexOf('bundlelive') === -1) {
          resp.clone().json().then(function(data) {
            var s = JSON.stringify(data).toLowerCase();
            if (s.indexOf('buyer') > -1 && (s.indexOf('price') > -1 || s.indexOf('cents') > -1 || s.indexOf('won') > -1)) {
              JSON.stringify(data, function(k, v) {
                if (v && typeof v === 'object' && v.buyer && (v.priceCents || v.totalCents || v.price)) {
                  var buyerName = (typeof v.buyer === 'object') ? v.buyer.username : v.buyer;
                  if (buyerName && !skipBuyers.test(buyerName)) {
                    sendSale(buyerName, (v.priceCents || v.totalCents || 0) / 100, (v.listing && v.listing.title) || v.title || 'Live Sale', 'gql');
                  }
                }
                return v;
              });
            }
          }).catch(function(){});
        }
      } catch(ex) {}
      return resp;
    });
  };

  // ─── STREAM INTEL ───────────────────────────────────
  setTimeout(function() {
    var intel = {};
    try {
      var titleMatch = document.title.match(/^(.+?)\s+is\s+live/i);
      if (titleMatch) intel.seller_username = titleMatch[1].trim();
      var urlMatch = window.location.pathname.match(/\/live\/([a-f0-9-]+)/i);
      if (urlMatch) intel.stream_id = urlMatch[1];
    } catch(e) {}
    if (Object.keys(intel).length) {
      fetch(SERVER + '/api/stream-intel', {
        method: 'POST', headers: authHeaders(), body: JSON.stringify(intel)
      }).catch(function() {});
    }
  }, 3000);

  console.log('[BL] BundleLive v3.8.0 - capture active (SmartChat:' + smartChatOn + ' AutoPrint:' + autoPrintOn + ' UserStopped:' + userStopped + ')');
})();
