/**
 * Generate BundleLive extension icons as PNG files
 * Uses pure Node.js — creates simple but recognizable icons
 * Run: node generate-icons.js
 */

// We'll create SVG and convert with a simple approach
// Since we can't use canvas easily, generate SVG files that Chrome can use
// Actually Chrome needs PNG. Let's create a simple HTML page that generates them.

const fs = require('fs');
const path = require('path');

// Create an HTML file that renders the icons on canvas and downloads them
const html = `<!DOCTYPE html>
<html><head><style>body{background:#000;display:flex;gap:20px;padding:20px;flex-wrap:wrap}</style></head>
<body>
<script>
const sizes = [16, 32, 48, 128];

sizes.forEach(size => {
  const canvas = document.createElement('canvas');
  canvas.width = size;
  canvas.height = size;
  const ctx = canvas.getContext('2d');
  
  // Background - rounded rect
  const r = size * 0.18;
  ctx.beginPath();
  ctx.roundRect(0, 0, size, size, r);
  ctx.fillStyle = '#ff4500';
  ctx.fill();
  
  // Inner darker area
  const p = size * 0.08;
  ctx.beginPath();
  ctx.roundRect(p, p, size - p*2, size - p*2, r * 0.7);
  ctx.fillStyle = '#e03e00';
  ctx.fill();
  
  // Lightning bolt ⚡
  ctx.fillStyle = '#ffffff';
  ctx.beginPath();
  const cx = size / 2;
  const cy = size / 2;
  const s = size * 0.35;
  
  // Simple bolt shape
  ctx.moveTo(cx + s*0.1, cy - s);      // top right
  ctx.lineTo(cx - s*0.35, cy + s*0.05); // mid left
  ctx.lineTo(cx + s*0.05, cy + s*0.05); // mid center
  ctx.lineTo(cx - s*0.1, cy + s);       // bottom left
  ctx.lineTo(cx + s*0.35, cy - s*0.05); // mid right  
  ctx.lineTo(cx - s*0.05, cy - s*0.05); // mid center
  ctx.closePath();
  ctx.fill();
  
  document.body.appendChild(canvas);
  
  // Auto-download
  const a = document.createElement('a');
  a.download = 'icon' + size + '.png';
  a.href = canvas.toDataURL('image/png');
  a.textContent = size + 'px';
  a.style.color = '#fff';
  a.style.display = 'block';
  document.body.appendChild(a);
});

// Also create the store icon (440x280 for Chrome Web Store promotional tile)
</script>
</body></html>`;

fs.writeFileSync(path.join(__dirname, 'generate-icons.html'), html);
console.log('Created generate-icons.html — open in browser to generate PNG icons');
console.log('Or use the SVG approach below...');

// Actually, let's just create simple SVG icons that we can reference
// Chrome extensions do support SVG in some contexts but PNGs are preferred
// Let's create a minimal PNG manually using raw bytes

// For now, create placeholder SVGs and note that real PNGs need browser or sharp
const svgTemplate = (size) => `<svg xmlns="http://www.w3.org/2000/svg" width="${size}" height="${size}" viewBox="0 0 ${size} ${size}">
  <rect width="${size}" height="${size}" rx="${size*0.18}" fill="#ff4500"/>
  <rect x="${size*0.08}" y="${size*0.08}" width="${size*0.84}" height="${size*0.84}" rx="${size*0.12}" fill="#e03e00"/>
  <text x="${size/2}" y="${size*0.68}" text-anchor="middle" font-size="${size*0.5}" font-weight="900" font-family="Arial" fill="white">⚡</text>
</svg>`;

[16, 32, 48, 128].forEach(size => {
  fs.writeFileSync(path.join(__dirname, `icon${size}.svg`), svgTemplate(size));
  console.log(`Created icon${size}.svg`);
});

console.log('\nTo convert to PNG, open generate-icons.html in Chrome and right-click save each canvas.');
console.log('Or install sharp: npm i sharp, then run convert script.');
