#!/bin/bash
# BundleLive Silent Printer — One-time install for Mac
# Uses Python 3 (built into macOS) — no Node.js needed

set -e

INSTALL_DIR="$HOME/.bundlelive"
PLIST_NAME="com.bundlelive.printserver"
PLIST_PATH="$HOME/Library/LaunchAgents/$PLIST_NAME.plist"
LOG_DIR="$HOME/Library/Logs/BundleLive"
SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"

echo ""
echo "🖨️  BundleLive Silent Printer Installer"
echo "========================================="
echo ""

# Find Python 3
PY=""
for p in python3 /usr/bin/python3 /Library/Frameworks/Python.framework/Versions/*/bin/python3; do
  if command -v "$p" &>/dev/null || [ -x "$p" ]; then
    PY="$p"
    break
  fi
done

if [ -z "$PY" ]; then
  echo "⚠️  Python 3 not found."
  echo "A popup may appear to install Command Line Tools — click Install."
  echo "After that finishes, run this script again."
  echo ""
  # Trigger the install dialog
  python3 --version 2>/dev/null || true
  exit 1
fi

echo "✓ Python 3 found: $PY ($($PY --version 2>&1))"

# Create directories
mkdir -p "$INSTALL_DIR"
mkdir -p "$LOG_DIR"

# Download or copy the Python print server
if [ -f "$SCRIPT_DIR/print-server.py" ]; then
  cp "$SCRIPT_DIR/print-server.py" "$INSTALL_DIR/print-server.py"
elif [ -f "$SCRIPT_DIR/../print-server.py" ]; then
  cp "$SCRIPT_DIR/../print-server.py" "$INSTALL_DIR/print-server.py"
else
  echo "Downloading print server..."
  curl -sL -o "$INSTALL_DIR/print-server.py" "https://bundlelive.com/downloads/print-server.py"
fi

echo "✓ Print server installed to $INSTALL_DIR"

# Stop existing service
launchctl unload "$PLIST_PATH" 2>/dev/null || true

# Create LaunchAgent
cat > "$PLIST_PATH" << EOF
<?xml version="1.0" encoding="UTF-8"?>
<!DOCTYPE plist PUBLIC "-//Apple//DTD PLIST 1.0//EN" "http://www.apple.com/DTDs/PropertyList-1.0.dtd">
<plist version="1.0">
<dict>
    <key>Label</key>
    <string>$PLIST_NAME</string>
    <key>ProgramArguments</key>
    <array>
        <string>$PY</string>
        <string>$INSTALL_DIR/print-server.py</string>
    </array>
    <key>RunAtLoad</key>
    <true/>
    <key>KeepAlive</key>
    <true/>
    <key>StandardOutPath</key>
    <string>$LOG_DIR/print-server.log</string>
    <key>StandardErrorPath</key>
    <string>$LOG_DIR/print-server-error.log</string>
</dict>
</plist>
EOF

launchctl load "$PLIST_PATH"
echo "✓ Service started"

sleep 2

if curl -s http://localhost:9099/health 2>/dev/null | grep -q "ok"; then
  echo ""
  echo "✅ SUCCESS! Silent printing is active."
  echo "   Bin tags print automatically — no dialogs"
  echo "   Auto-starts on login"
  echo ""
else
  echo ""
  echo "⚠️  Service started but health check failed."
  echo "   Check: cat $LOG_DIR/print-server-error.log"
fi
