/**
 * BundleLive Extension — Orders Page Label Sync
 * Auto-runs on whatnot.com/orders* pages.
 * Scrapes order data and matches with BundleLive bins for quick label generation.
 */
(function() {
  'use strict';
  if (window.__bundlelive_orders) return;
  window.__bundlelive_orders = true;

  let serverUrl = 'https://bundlelive.com';
  let authToken = '';

  // Load server URL and auth token
  if (typeof chrome !== 'undefined' && chrome.storage) {
    chrome.storage.local.get(['serverUrl', 'authToken'], (result) => {
      if (result.serverUrl) serverUrl = result.serverUrl;
      if (result.authToken) authToken = result.authToken;
      init();
    });
  } else {
    init();
  }

  function authHeaders() {
    var h = { 'Content-Type': 'application/json' };
    if (authToken) h['Authorization'] = 'Bearer ' + authToken;
    return h;
  }

  function init() {
    console.log('[BL Orders] ⚡ BundleLive Orders Sync loaded');
    injectPanel();
  }

  // ─── INJECT FLOATING PANEL ──────────────────────────
  function injectPanel() {
    const panel = document.createElement('div');
    panel.id = 'bl-orders-panel';
    panel.innerHTML = `
      <div style="position:fixed;bottom:20px;right:20px;z-index:99999;background:#0d0d1a;border:2px solid #ff4500;border-radius:16px;padding:20px;width:360px;max-height:500px;overflow-y:auto;box-shadow:0 8px 40px rgba(0,0,0,0.6);font-family:system-ui,sans-serif;color:#fff;">
        <div style="display:flex;justify-content:space-between;align-items:center;margin-bottom:12px;">
          <h3 style="margin:0;font-size:16px;">⚡ BundleLive — Ship & Pack</h3>
          <button id="bl-close" style="background:none;border:none;color:#71717a;font-size:18px;cursor:pointer;">✕</button>
        </div>
        <div id="bl-orders-status" style="color:#a1a1aa;font-size:13px;margin-bottom:12px;">Scanning orders on this page...</div>
        <div id="bl-orders-list" style="font-size:13px;"></div>
        <div style="margin-top:12px;display:flex;gap:8px;">
          <button id="bl-scan" style="flex:1;padding:10px;background:#ff4500;color:#fff;border:none;border-radius:8px;font-weight:600;cursor:pointer;font-size:13px;">🔍 Scan Orders</button>
          <button id="bl-match" style="flex:1;padding:10px;background:#1a1a2e;color:#fff;border:1px solid #27272a;border-radius:8px;font-weight:600;cursor:pointer;font-size:13px;">📦 Match Bins</button>
        </div>
      </div>
    `;
    document.body.appendChild(panel);

    document.getElementById('bl-close').onclick = () => panel.remove();
    document.getElementById('bl-scan').onclick = scanOrders;
    document.getElementById('bl-match').onclick = matchBins;
  }

  // ─── SCAN ORDERS FROM PAGE ──────────────────────────
  let scannedOrders = [];

  function scanOrders() {
    const status = document.getElementById('bl-orders-status');
    const list = document.getElementById('bl-orders-list');
    status.textContent = 'Scanning...';
    scannedOrders = [];

    try {
      // Strategy 1: Look for order rows/cards on the page
      // Whatnot orders page structure varies — try multiple selectors
      const orderElements = document.querySelectorAll('[class*="order"], [class*="Order"], tr, [data-testid*="order"]');
      
      for (const el of orderElements) {
        const text = el.textContent || '';
        // Look for username + price patterns
        const userMatch = text.match(/@(\w{2,30})/);
        const priceMatch = text.match(/\$(\d+(?:\.\d{2})?)/);
        
        if (userMatch && priceMatch) {
          scannedOrders.push({
            username: userMatch[1],
            price: parseFloat(priceMatch[1]),
            text: text.substring(0, 100).trim(),
            element: el
          });
        }
      }

      // Strategy 2: Scan all visible text for "username" + "$price" patterns
      if (scannedOrders.length === 0) {
        const allText = document.body.innerText;
        const lines = allText.split('\n');
        for (const line of lines) {
          const userMatch = line.match(/@?(\w{3,30})/);
          const priceMatch = line.match(/\$(\d+(?:\.\d{2})?)/);
          if (userMatch && priceMatch && parseFloat(priceMatch[1]) > 0) {
            scannedOrders.push({
              username: userMatch[1],
              price: parseFloat(priceMatch[1]),
              text: line.trim().substring(0, 100)
            });
          }
        }
      }

      // Deduplicate by username
      const unique = {};
      for (const order of scannedOrders) {
        if (!unique[order.username]) {
          unique[order.username] = { ...order, count: 1, total: order.price };
        } else {
          unique[order.username].count++;
          unique[order.username].total += order.price;
        }
      }

      const grouped = Object.values(unique);
      status.textContent = `Found ${scannedOrders.length} orders from ${grouped.length} buyers`;

      list.innerHTML = grouped.map(o => `
        <div style="display:flex;justify-content:space-between;padding:8px 0;border-bottom:1px solid #1e1e2e;">
          <span>@${o.username} <span style="color:#71717a;">(${o.count} items)</span></span>
          <span style="color:#ff4500;font-weight:600;">$${o.total.toFixed(2)}</span>
        </div>
      `).join('');

    } catch(e) {
      status.textContent = 'Scan error: ' + e.message;
      console.error('[BL Orders] Scan error:', e);
    }
  }

  // ─── MATCH WITH BUNDLELIVE BINS ─────────────────────
  async function matchBins() {
    const status = document.getElementById('bl-orders-status');
    const list = document.getElementById('bl-orders-list');
    
    if (scannedOrders.length === 0) {
      status.textContent = 'Scan orders first!';
      return;
    }

    if (!authToken) {
      status.textContent = '⚠️ Not logged in. Open the BundleLive extension popup and log in first.';
      return;
    }

    status.textContent = 'Fetching bin data from BundleLive...';

    try {
      const res = await fetch(serverUrl + '/api/shows/active', { headers: authHeaders() });
      const showData = await res.json();

      if (!showData.show && !showData.bins) {
        // Try getting latest ended show
        const histRes = await fetch(serverUrl + '/api/shows/history', { headers: authHeaders() });
        const histData = await histRes.json();
        
        if (histData.shows && histData.shows.length > 0) {
          status.textContent = 'Using latest show data...';
          // Use most recent show
        } else {
          status.textContent = '⚠️ No show data found. Run a show first.';
          return;
        }
      }

      const bins = showData.bins || {};
      
      // Match orders to bins
      const matched = [];
      const unmatched = [];
      
      const uniqueBuyers = {};
      for (const o of scannedOrders) {
        if (!uniqueBuyers[o.username]) uniqueBuyers[o.username] = { ...o, count: 1, total: o.price };
        else { uniqueBuyers[o.username].count++; uniqueBuyers[o.username].total += o.price; }
      }

      for (const [username, orderInfo] of Object.entries(uniqueBuyers)) {
        const bin = Object.values(bins).find(b => 
          b.buyer_username?.toLowerCase() === username.toLowerCase()
        );
        
        if (bin) {
          matched.push({ ...orderInfo, bin: bin.label || bin.bin_number, binId: bin.id });
        } else {
          unmatched.push(orderInfo);
        }
      }

      status.textContent = `Matched ${matched.length}/${Object.keys(uniqueBuyers).length} buyers to bins`;

      list.innerHTML = 
        '<div style="font-weight:600;color:#4ade80;margin:8px 0;">✅ Matched:</div>' +
        matched.map(m => `
          <div style="display:flex;justify-content:space-between;padding:6px 0;border-bottom:1px solid #1e1e2e;">
            <span>📦 Bin #${m.bin} — @${m.username} (${m.count} items)</span>
            <span style="color:#ff4500;">$${m.total.toFixed(2)}</span>
          </div>
        `).join('') +
        (unmatched.length ? 
          '<div style="font-weight:600;color:#fbbf24;margin:8px 0;">⚠️ No bin match:</div>' +
          unmatched.map(u => `
            <div style="display:flex;justify-content:space-between;padding:6px 0;border-bottom:1px solid #1e1e2e;">
              <span>@${u.username} (${u.count} items)</span>
              <span style="color:#71717a;">$${u.total.toFixed(2)}</span>
            </div>
          `).join('') 
        : '');

    } catch(e) {
      status.textContent = 'Error connecting to BundleLive: ' + e.message;
      console.error('[BL Orders] Match error:', e);
    }
  }

  // Listen for URL changes (SPA navigation)
  if (typeof chrome !== 'undefined' && chrome.storage) {
    chrome.storage.onChanged.addListener((changes) => {
      if (changes.serverUrl) serverUrl = changes.serverUrl.newValue;
      if (changes.authToken) authToken = changes.authToken.newValue || '';
    });
  }
})();
