#!/usr/bin/env node
/**
 * BundleLive Silent Print Server v2.0
 * Generates PDF bin tags and sends to default printer via lp (Mac) 
 * No HTML rendering needed — creates PDF directly with raw PDF commands
 */

const http = require('http');
const fs = require('fs');
const path = require('path');
const os = require('os');
const { execSync } = require('child_process');

const PORT = 9099;

// Generate a minimal PDF with bin number and buyer name
// No external dependencies — pure PDF spec
function generateBinTagPDF(binLabel, buyerName) {
  // Page size: 4x2 inches = 288x144 points
  const W = 288, H = 144;
  
  const content = [
    'BT',
    '/F1 48 Tf',           // Font size 48 for bin number
    `72 80 Td`,            // Position
    `(${binLabel}) Tj`,    // Bin label text
    'ET',
    'BT',
    '/F1 18 Tf',           // Font size 18 for buyer name
    `72 40 Td`,            // Position below
    `(@${buyerName}) Tj`,  // Buyer name
    'ET'
  ].join('\n');

  const stream = `stream\n${content}\nendstream`;
  const streamLen = content.length;

  const pdf = `%PDF-1.4
1 0 obj
<< /Type /Catalog /Pages 2 0 R >>
endobj

2 0 obj
<< /Type /Pages /Kids [3 0 R] /Count 1 >>
endobj

3 0 obj
<< /Type /Page /Parent 2 0 R /MediaBox [0 0 ${W} ${H}] /Contents 4 0 R /Resources << /Font << /F1 5 0 R >> >> >>
endobj

4 0 obj
<< /Length ${streamLen} >>
${stream}
endobj

5 0 obj
<< /Type /Font /Subtype /Type1 /BaseFont /Helvetica-Bold >>
endobj

xref
0 6
0000000000 65535 f 
0000000009 00000 n 
0000000058 00000 n 
0000000115 00000 n 
0000000266 00000 n 
0000000${(317 + streamLen).toString().padStart(3, '0')} 00000 n 

trailer
<< /Size 6 /Root 1 0 R >>
startxref
0
%%EOF`;

  return pdf;
}

// Alternative: use textutil (Mac built-in) to convert HTML to PDF
function htmlToPdfMac(htmlFile, pdfFile) {
  try {
    // Try textutil first (built into macOS)
    execSync(`textutil -convert pdf "${htmlFile}" -output "${pdfFile}"`, { timeout: 10000 });
    return true;
  } catch(e) {
    try {
      // Try cupsfilter as backup
      execSync(`/usr/sbin/cupsfilter -m application/pdf "${htmlFile}" > "${pdfFile}" 2>/dev/null`, { timeout: 10000 });
      return true;
    } catch(e2) {
      return false;
    }
  }
}

const server = http.createServer((req, res) => {
  res.setHeader('Access-Control-Allow-Origin', '*');
  res.setHeader('Access-Control-Allow-Methods', 'POST, OPTIONS');
  res.setHeader('Access-Control-Allow-Headers', 'Content-Type');
  if (req.method === 'OPTIONS') { res.writeHead(204); return res.end(); }

  if (req.method === 'POST' && req.url === '/print') {
    let body = '';
    req.on('data', chunk => body += chunk);
    req.on('end', () => {
      try {
        const data = JSON.parse(body);
        const { html, width, height, binLabel, buyerName } = data;
        
        const tmpBase = path.join(os.tmpdir(), `bl-label-${Date.now()}`);
        const platform = os.platform();
        let printed = false;

        // Method 1: If binLabel and buyerName provided, generate PDF directly (best quality)
        if (binLabel && buyerName) {
          const pdfFile = tmpBase + '.pdf';
          const pdfContent = generateBinTagPDF(binLabel, buyerName);
          fs.writeFileSync(pdfFile, pdfContent);
          
          try {
            if (platform === 'darwin' || platform === 'linux') {
              execSync(`lp "${pdfFile}"`, { timeout: 10000 });
            } else if (platform === 'win32') {
              execSync(`start /min AcroRd32.exe /t "${pdfFile}"`, { timeout: 10000 });
            }
            printed = true;
            console.log(`[PrintServer] ✅ PDF bin tag printed: ${binLabel} @${buyerName}`);
            setTimeout(() => { try { fs.unlinkSync(pdfFile); } catch(e) {} }, 30000);
          } catch(e) {
            console.error('[PrintServer] PDF print failed:', e.message);
          }
        }

        // Method 2: HTML provided — convert to PDF then print (Mac)
        if (!printed && html) {
          const htmlFile = tmpBase + '.html';
          const pdfFile = tmpBase + '-converted.pdf';
          
          const fullHtml = `<!DOCTYPE html><html><head><style>
@page { size: ${width || '4in'} ${height || '2in'}; margin: 0; }
html, body { margin: 0; padding: 0; }
</style></head><body>${html}</body></html>`;
          
          fs.writeFileSync(htmlFile, fullHtml);

          if (platform === 'darwin') {
            // Try converting HTML to PDF on Mac
            if (htmlToPdfMac(htmlFile, pdfFile) && fs.existsSync(pdfFile)) {
              try {
                execSync(`lp "${pdfFile}"`, { timeout: 10000 });
                printed = true;
                console.log(`[PrintServer] ✅ HTML→PDF printed`);
              } catch(e) {
                console.error('[PrintServer] lp failed:', e.message);
              }
              setTimeout(() => { try { fs.unlinkSync(pdfFile); } catch(e) {} }, 30000);
            }
            
            // Fallback: try lp with HTML directly (some CUPS configs handle it)
            if (!printed) {
              try {
                execSync(`lp -o media=Custom.${width || '4in'}x${height || '2in'} "${htmlFile}"`, { timeout: 10000 });
                printed = true;
                console.log(`[PrintServer] ✅ HTML printed directly via lp`);
              } catch(e) {
                console.error('[PrintServer] Direct HTML print failed:', e.message);
              }
            }
          }
          
          setTimeout(() => { try { fs.unlinkSync(htmlFile); } catch(e) {} }, 30000);
        }

        if (printed) {
          res.writeHead(200, { 'Content-Type': 'application/json' });
          res.end(JSON.stringify({ success: true }));
        } else {
          res.writeHead(500, { 'Content-Type': 'application/json' });
          res.end(JSON.stringify({ success: false, error: 'All print methods failed' }));
        }

      } catch (e) {
        res.writeHead(400, { 'Content-Type': 'application/json' });
        res.end(JSON.stringify({ success: false, error: e.message }));
      }
    });
  } else if (req.method === 'GET' && req.url === '/health') {
    res.writeHead(200, { 'Content-Type': 'application/json' });
    res.end(JSON.stringify({ status: 'ok', version: '2.0.0' }));
  } else {
    res.writeHead(404);
    res.end('Not found');
  }
});

server.listen(PORT, '127.0.0.1', () => {
  console.log(`\n🖨️  BundleLive Print Server v2.0 running on http://localhost:${PORT}`);
  console.log(`   Platform: ${os.platform()}`);
  console.log(`   Waiting for print jobs...\n`);
});
