#!/usr/bin/env python3
"""
BundleLive Silent Print Server v1.0
Pure Python — no dependencies needed. Works on any Mac with python3.
Listens on port 9099, receives print jobs, generates PDF, sends to default printer.
"""

import http.server
import json
import os
import subprocess
import tempfile
import time
import struct

PORT = 9099

def generate_pdf(bin_label, buyer_name):
    """Generate a minimal PDF bin tag. No external libraries needed."""
    # Simple PDF with Helvetica-Bold text
    content_stream = f"""BT
/F1 60 Tf
50 85 Td
({bin_label}) Tj
ET
BT
/F1 22 Tf
50 40 Td
(@{buyer_name}) Tj
ET"""
    
    stream_bytes = content_stream.encode('latin-1')
    
    objects = []
    offsets = []
    pdf = b""
    
    # Header
    pdf += b"%PDF-1.4\n"
    
    # Object 1: Catalog
    offsets.append(len(pdf))
    pdf += b"1 0 obj\n<< /Type /Catalog /Pages 2 0 R >>\nendobj\n"
    
    # Object 2: Pages
    offsets.append(len(pdf))
    pdf += b"2 0 obj\n<< /Type /Pages /Kids [3 0 R] /Count 1 >>\nendobj\n"
    
    # Object 3: Page (4x2 inches = 288x144 points)
    offsets.append(len(pdf))
    pdf += b"3 0 obj\n<< /Type /Page /Parent 2 0 R /MediaBox [0 0 288 144] /Contents 4 0 R /Resources << /Font << /F1 5 0 R >> >> >>\nendobj\n"
    
    # Object 4: Content stream
    offsets.append(len(pdf))
    pdf += f"4 0 obj\n<< /Length {len(stream_bytes)} >>\nstream\n".encode('latin-1')
    pdf += stream_bytes
    pdf += b"\nendstream\nendobj\n"
    
    # Object 5: Font
    offsets.append(len(pdf))
    pdf += b"5 0 obj\n<< /Type /Font /Subtype /Type1 /BaseFont /Helvetica-Bold >>\nendobj\n"
    
    # Cross-reference table
    xref_offset = len(pdf)
    pdf += b"xref\n"
    pdf += b"0 6\n"
    pdf += b"0000000000 65535 f \n"
    for off in offsets:
        pdf += f"{off:010d} 00000 n \n".encode('latin-1')
    
    # Trailer
    pdf += b"trailer\n<< /Size 6 /Root 1 0 R >>\n"
    pdf += f"startxref\n{xref_offset}\n%%EOF\n".encode('latin-1')
    
    return pdf


class PrintHandler(http.server.BaseHTTPRequestHandler):
    def log_message(self, format, *args):
        print(f"[PrintServer] {args[0]}" if args else "")
    
    def send_cors_headers(self):
        self.send_header('Access-Control-Allow-Origin', '*')
        self.send_header('Access-Control-Allow-Methods', 'POST, OPTIONS, GET')
        self.send_header('Access-Control-Allow-Headers', 'Content-Type')
    
    def do_OPTIONS(self):
        self.send_response(204)
        self.send_cors_headers()
        self.end_headers()
    
    def do_GET(self):
        if self.path == '/health':
            self.send_response(200)
            self.send_header('Content-Type', 'application/json')
            self.send_cors_headers()
            self.end_headers()
            self.wfile.write(json.dumps({"status": "ok", "version": "1.0.0", "engine": "python"}).encode())
        else:
            self.send_response(404)
            self.end_headers()
    
    def do_POST(self):
        if self.path == '/print':
            content_length = int(self.headers.get('Content-Length', 0))
            body = self.rfile.read(content_length)
            
            try:
                data = json.loads(body)
                bin_label = data.get('binLabel', '')
                buyer_name = data.get('buyerName', '')
                
                if not bin_label and not buyer_name:
                    # Try to extract from HTML if provided
                    html = data.get('html', '')
                    if html:
                        # Just print a generic label
                        bin_label = 'BIN'
                        buyer_name = 'label'
                
                if bin_label or buyer_name:
                    # Generate PDF
                    pdf_data = generate_pdf(bin_label, buyer_name)
                    
                    # Write to temp file
                    tmp = tempfile.NamedTemporaryFile(suffix='.pdf', delete=False, prefix='bl-label-')
                    tmp.write(pdf_data)
                    tmp.close()
                    
                    # Print via lp
                    try:
                        result = subprocess.run(
                            ['lp', tmp.name],
                            capture_output=True, text=True, timeout=10
                        )
                        if result.returncode == 0:
                            print(f"[PrintServer] ✅ Printed: {bin_label} @{buyer_name}")
                            self.send_response(200)
                            self.send_header('Content-Type', 'application/json')
                            self.send_cors_headers()
                            self.end_headers()
                            self.wfile.write(json.dumps({"success": True}).encode())
                        else:
                            error = result.stderr.strip()
                            print(f"[PrintServer] ❌ lp error: {error}")
                            self.send_response(500)
                            self.send_header('Content-Type', 'application/json')
                            self.send_cors_headers()
                            self.end_headers()
                            self.wfile.write(json.dumps({"success": False, "error": error}).encode())
                    except Exception as e:
                        print(f"[PrintServer] ❌ Print error: {e}")
                        self.send_response(500)
                        self.send_header('Content-Type', 'application/json')
                        self.send_cors_headers()
                        self.end_headers()
                        self.wfile.write(json.dumps({"success": False, "error": str(e)}).encode())
                    
                    # Clean up after delay
                    try:
                        time.sleep(1)
                        os.unlink(tmp.name)
                    except:
                        pass
                else:
                    self.send_response(400)
                    self.send_header('Content-Type', 'application/json')
                    self.send_cors_headers()
                    self.end_headers()
                    self.wfile.write(json.dumps({"success": False, "error": "No binLabel or buyerName"}).encode())
                    
            except json.JSONDecodeError as e:
                self.send_response(400)
                self.send_header('Content-Type', 'application/json')
                self.send_cors_headers()
                self.end_headers()
                self.wfile.write(json.dumps({"success": False, "error": f"Invalid JSON: {e}"}).encode())
        else:
            self.send_response(404)
            self.end_headers()


if __name__ == '__main__':
    server = http.server.HTTPServer(('127.0.0.1', PORT), PrintHandler)
    print(f"\n🖨️  BundleLive Print Server v1.0 (Python)")
    print(f"   Running on http://localhost:{PORT}")
    print(f"   Waiting for print jobs...\n")
    try:
        server.serve_forever()
    except KeyboardInterrupt:
        print("\n[PrintServer] Stopped.")
        server.server_close()
