const DEFAULT_URL = 'https://bundlelive.com';
const loginView = document.getElementById('login-view');
const mainView = document.getElementById('main-view');

// Check if logged in
chrome.storage.local.get(['authToken', 'serverUrl', 'userEmail', 'smartChatEnabled', 'autoPrintEnabled'], (r) => {
  if (r.authToken) {
    showMainView(r);
  } else {
    showLoginView();
  }
});

// ─── LOGIN ────────────────────────────────────────────
function showLoginView() {
  loginView.style.display = 'block';
  mainView.style.display = 'none';
}

document.getElementById('login-btn').addEventListener('click', doLogin);
document.getElementById('login-password').addEventListener('keydown', (e) => {
  if (e.key === 'Enter') doLogin();
});

async function doLogin() {
  const email = document.getElementById('login-email').value.trim();
  const password = document.getElementById('login-password').value;
  const errorEl = document.getElementById('login-error');
  const btn = document.getElementById('login-btn');
  
  if (!email || !password) {
    errorEl.textContent = 'Enter email and password';
    errorEl.style.display = 'block';
    return;
  }

  btn.textContent = 'Logging in...';
  btn.disabled = true;
  errorEl.style.display = 'none';

  try {
    const serverUrl = DEFAULT_URL;
    const resp = await fetch(serverUrl + '/api/auth/login', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ email, password })
    });
    const data = await resp.json();

    if (data.success && data.token) {
      // Save auth
      chrome.storage.local.set({
        authToken: data.token,
        serverUrl: serverUrl,
        userEmail: email
      });
      showMainView({ authToken: data.token, serverUrl, userEmail: email });
    } else {
      errorEl.textContent = data.error || 'Login failed';
      errorEl.style.display = 'block';
    }
  } catch (e) {
    errorEl.textContent = 'Cannot reach server';
    errorEl.style.display = 'block';
  }

  btn.textContent = 'Log In';
  btn.disabled = false;
}

// ─── MAIN VIEW ────────────────────────────────────────
function showMainView(r) {
  loginView.style.display = 'none';
  mainView.style.display = 'block';

  const serverUrl = r.serverUrl || DEFAULT_URL;
  if (r.userEmail) {
    document.getElementById('user-info').textContent = 'Logged in as ' + r.userEmail;
  }

  // Load toggles
  document.getElementById('chatToggle').checked = !!r.smartChatEnabled;
  document.getElementById('printToggle').checked = !!r.autoPrintEnabled;

  // Check connection
  checkConnection(serverUrl, r.authToken);

  // Toggle handlers
  document.getElementById('chatToggle').addEventListener('change', (e) => {
    chrome.storage.local.set({ smartChatEnabled: e.target.checked });
  });
  document.getElementById('printToggle').addEventListener('change', (e) => {
    chrome.storage.local.set({ autoPrintEnabled: e.target.checked });
  });

  // Dashboard button
  document.getElementById('dashBtn').addEventListener('click', () => {
    chrome.tabs.create({ url: serverUrl + '/app' });
  });

  // Clear button — full reset of ALL state
  document.getElementById('clearBtn').addEventListener('click', () => {
    chrome.storage.local.set({
      salesCount: 0, totalRevenue: 0, uniqueBuyers: 0,
      uniqueBuyersList: [], sentSaleKeys: {}
    });
    chrome.runtime.sendMessage({ type: 'clear_sales' });
    // Tell capture.js to wipe its in-memory dedup too
    chrome.tabs.query({ url: '*://*.whatnot.com/*' }, (tabs) => {
      tabs.forEach(tab => {
        chrome.tabs.sendMessage(tab.id, { type: 'clear_all_state' }).catch(() => {});
      });
    });
    document.getElementById('salesCount').textContent = '0';
    const revEl = document.getElementById('revenueDisplay');
    if (revEl) revEl.textContent = '$0';
    const binsEl = document.getElementById('binsCount');
    if (binsEl) binsEl.textContent = '0';
    document.getElementById('showStatus').innerHTML = '<span class="show-badge off">OFF</span>';
  });

  // Logout button
  document.getElementById('logoutBtn').addEventListener('click', () => {
    chrome.storage.local.remove(['authToken', 'userEmail']);
    showLoginView();
  });

  // Check for update notification
  chrome.storage.local.get(['extension_update_available', 'extension_latest_version'], (upd) => {
    if (upd.extension_update_available && upd.extension_latest_version) {
      const bar = document.getElementById('update-bar');
      bar.textContent = '🔄 Update available: v' + upd.extension_latest_version + ' — visit bundlelive.com to update';
      bar.style.display = 'block';
    }
  });

  // Get sales count and revenue from storage (synced by capture.js)
  chrome.storage.local.get(['salesCount', 'totalRevenue'], (data) => {
    if (data.salesCount) {
      document.getElementById('salesCount').textContent = data.salesCount;
    }
    if (data.totalRevenue) {
      const revenueEl = document.getElementById('revenueDisplay');
      if (revenueEl) revenueEl.textContent = '$' + Number(data.totalRevenue).toFixed(0);
    }
  });
  // Also try background
  chrome.runtime.sendMessage({ type: 'get_total' }, (response) => {
    if (response && response.totalSales) {
      document.getElementById('salesCount').textContent = response.totalSales;
    }
  });
  // Live update when capture.js writes new count
  chrome.storage.onChanged.addListener((changes) => {
    if (changes.salesCount) document.getElementById('salesCount').textContent = changes.salesCount.newValue;
    if (changes.totalRevenue) {
      const revenueEl = document.getElementById('revenueDisplay');
      if (revenueEl) revenueEl.textContent = '$' + Number(changes.totalRevenue.newValue).toFixed(0);
    }
  });
}

async function checkConnection(url, token) {
  const dot = document.getElementById('statusDot');
  const text = document.getElementById('statusText');
  const binsEl = document.getElementById('binsCount');
  const showEl = document.getElementById('showStatus');

  const headers = { 'Content-Type': 'application/json' };
  if (token) headers['Authorization'] = 'Bearer ' + token;

  try {
    const resp = await fetch(url + '/api/shows/active', { headers });
    const data = await resp.json();
    if (data.success && data.show) {
      dot.className = 'dot green';
      text.textContent = 'Connected — show active';
      showEl.innerHTML = '<span class="show-badge live">LIVE</span>';
      const binCount = data.bins ? data.bins.length : 0;
      binsEl.textContent = binCount;
    } else {
      dot.className = 'dot green';
      text.textContent = 'Connected — no active show';
      showEl.innerHTML = '<span class="show-badge off">OFF</span>';
      binsEl.textContent = '—';
    }
  } catch (e) {
    dot.className = 'dot red';
    text.textContent = 'Cannot reach server';
    binsEl.textContent = '—';
  }
}
